/**
   \file pardiso.h
   \brief Pardiso interface
   
   This is a facade code which enables and hopefully, simplifies
   the access from within C to the pardiso 
   code by Olaf Schenk and Klaus Gaertner.
*/
#ifndef PARDISO_H
#define PARDISO_H

#define PARDISO_VERSION 200001
/*--------------------------------------------------------------------*/
typedef void (*pardisoType)(
	      void * handle,
	      int *max_factorizations,
	      int *matrix_num,
	      int *matrix_type,
	      int *ido,
	      int *neqns,
	      double *a,
	      int *ia,
	      int *ja,
	      int *perm_user,
	      int *nb,
	      int iparam[64],
	      int *msglvl,
	      double *b,
	      double *x,
	      int *error
	      );

#
/* 

   Direct access to original pardiso, documentation see in pardiso_header.f90
   All array addresses are offset 0, i.e. if  array a is passed 
   to the code, a[0] is accessed as A(1) from within fortran.
   All indices in the  index fields are assumed to  count  from one.

*/


/*--------------------------------------------------------------------*/
/*Facade code for symbolic factorization part*/

typedef struct pardisoSym_struct *pardisoSym;


/** 
   Create an instance of a structure holding data 
   for the symbolic part of the pardiso code.
   The input is the symbolic part of the classical CRS structure.
 */
pardisoSym  pardisoSymCreate(int n, int nja, int *ia, int*ja);


pardisoSym  pardisoSymCreate2(int n, int nja, int *ia, int*ja,int complex, int strsym);

/** 
   Enter Information about space dimension. Heuristically,
   by this information, the re-ordering algorithm is chosen.
*/
void pardisoSymInfoSpacedim(pardisoSym psym, int dim);


/**
   Perform a symbolic factorization 
*/
void pardisoSymFactorize(pardisoSym psym);


/**
  Free all memory associated to the symbolic factorization. The ia
  and ja arrays are _not_ touched.
 */
void pardisoSymDestroy(pardisoSym p);


/*--------------------------------------------------------------------*/
/*Facade code for numerical  factorization  and solver part*/

typedef struct pardisoNum_struct *pardisoNum;

/**
  Create an instance of a full matrix information structure 
*/
pardisoNum pardisoNumCreate(pardisoSym psym);


/**
 Load array a into structure 
*/
void pardisoNumLoad(pardisoNum pnum, double *a);




/**
 Load complex array a into structure 
*/
void pardisoNumLoadComplex(pardisoNum pnum, double *a);

/** 
    Perform numerical factorization. Before, the symbolic factorization
    of the associated pardisoSym structure has to be called.
*/
void pardisoNumFactorize(pardisoNum pnum);


/** 
    Perform numerical factorization with pivoting. A  symbolic factorization
    of the associated pardisoSym structure will be ignored.
*/
void pardisoNumFactorizePivot(pardisoNum pnum);



/** 
    Destroy the factorization data associated to the numeric part of the problem, but not
    all of the work space created during the numerical factorization process  of pnum.
*/
void pardisoNumDestroy(pardisoNum pnum);

/**
   Solve the problem based with right hand side rhs  and 
   solution sol. Possibly do an iterative refinement using CGS.
*/
void pardisoNumSolve(pardisoNum p,double *sol, double *rhs);

/** 
    Solve the problem based with right hand side rhs and solution
    sol. Perform exactly one step of the LU backsubstitution. 
*/
void pardisoNumPrecon(pardisoNum p,double *u, double *rhs);


/*--------------------------------------------------------------------*/
/* Misc stuff */

/* Set the verbosity of the module */
void pardisoSetMsglevel(int ilev);

/** To initialize this module, it  needs to be given the symbol of the
   pardiso code from the pardiso fortran library.
   
   For static initialization, e.g. do the following:

   { 
   extern void pardiso_();
   pardisoInit((PardisoType)pardiso_);
   }

   For initializtion after loading pardiso by dlopen, you can submit
   the result returned by dlsym().
   */
void pardisoInit(pardisoType pardiso);


/**
   Check if the module already had been initialized.
*/
int pardisoInitialized(void);

#define PARDISO_MAX_NUM_PER_SYM 10
/* This is the maximal number of allowed numerical factorizations per symbolic one*/



struct pardisoSym_struct
{

  int n;
  int nja;
  int *ia;
  int *ja;

  void **handle;
  int handle_size;
  int matrix_type;
  int iparam[64];
  int max_factorizations;
  int used_matrix_numbers[PARDISO_MAX_NUM_PER_SYM];
  int factorized;
};

struct pardisoNum_struct
{
  double *a;
  int num;
  double *sol;
  int nrhs;
  int is_complex;
  pardisoSym psym;
  int matrix_type;
};

#endif

